// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "UObject/NoExportTypes.h"
#include "DynamicTextureUtilities.h"

#include "Tileset.generated.h"



USTRUCT()
struct FAsset
{
	GENERATED_USTRUCT_BODY()

		UPROPERTY()
		FString version;

	UPROPERTY()
		FString tilesetVersion;

	UPROPERTY()
		FString gltfUpAxis;

	FAsset() {
		version = "";
		tilesetVersion = "";
	}
};


USTRUCT()
struct FProperties
{
	GENERATED_USTRUCT_BODY()

		UPROPERTY()
		float maximum;

	UPROPERTY()
		float minimum;
};

USTRUCT()
struct FPropertieDict
{
	GENERATED_USTRUCT_BODY()
		//FIXME  should be a map or so, found no Unreal doc, values are from an example
		/*"patternProperties" : {
		".*": {
		"$ref": "properties.schema.json"
		}
		}*/

		UPROPERTY()
		FProperties Longitude;

	UPROPERTY()
		FProperties Latitude;

	UPROPERTY()
		FProperties Height;
};



USTRUCT()
struct FBoundingVolume
{
	GENERATED_USTRUCT_BODY()
		//TODO only one in Unreal

		UPROPERTY()
		TArray<float> box; //12 Elemete

	UPROPERTY()
		TArray<float> region; //6 Elemete

	UPROPERTY()
		TArray<float> sphere; //4 Elemete

	float getDistanceTo(FVector LocalPosition, FMatrix bouindingTransformation);
};

USTRUCT()
struct FTileContent
{
	GENERATED_USTRUCT_BODY()

		UPROPERTY()
		FBoundingVolume boundingVolume;

	UPROPERTY()
		FString url;

	UPROPERTY()
	bool loadingStarted;


	TArray<uint8> content;
	struct FTileset *tileset;
	TArray<class AProceduralEntity*> tiles;
	TArray<FMatrix> absoluteTileTranforms;

	FTileContent() {
		tileset = nullptr;
		loadingStarted = false;
	}

};



USTRUCT()
struct FTile
{
	GENERATED_USTRUCT_BODY()

		UPROPERTY()
		FBoundingVolume boundingVolume;

	UPROPERTY()
		FBoundingVolume viewerRequestVolume;

	UPROPERTY()
		float geometricError;

	UPROPERTY()
		FString refine; //TODO: "enum" : ["add", "replace"]

	UPROPERTY()
		TArray<float> transform; //16 Elemete
								 //"default" : [ 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0 ]

	UPROPERTY()
		FTileContent content;

	//UPROPERTY()
	//struct FTile children;
	TArray<struct FTile*> children;
	struct FTileset *parentTilset;
	struct FTile *parent;
	FMatrix *realtiveTransform;
	FMatrix *absoluteTransform;

	FMatrix *getMatrix();
	FMatrix *getAbsoluteTransform();

	double getScreenSpaceError(double lambda, FVector CamLocation);
	void setVisible(bool visible);






	FTile() {
		float transformDefault[] = { 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0 };
		transform.Empty();
		parentTilset = nullptr;
		parent = nullptr;
		realtiveTransform = nullptr;
		absoluteTransform = nullptr;
		//transform.Append(transformDefault, 16);
	}


};

USTRUCT()
struct FTileset
{
	GENERATED_USTRUCT_BODY()

		UPROPERTY()
		FAsset asset;

	UPROPERTY()
		FPropertieDict properties;

	UPROPERTY()
		float geometricError;

	UPROPERTY()
		FTile root;

	FString absoluteURL;
	FTile* parent;
	class ATilesetActor* parentActor;

	FTileset() {
		geometricError = -1;
		parent = nullptr;
		parentActor = nullptr;
		
	}
};

//3D Tiles

struct TileHeader {
	unsigned char magic[4];
	uint32 version;
	uint32 byteLength;

	FString getMagicAsFString() {
		char ansiiData[5]; //A temp buffer for the data
		memcpy(ansiiData, magic, 4); //Assumes bytesRead is always smaller than 1024 bytes
		ansiiData[4] = 0; //Add null terminator
		return ANSI_TO_TCHAR(ansiiData); //Convert to FString
	}

	bool isB3DM() {
		return  magic[0] == 'b' && magic[1] == '3' && magic[2] == 'd' && magic[3] == 'm';
	}
	bool isI3DM() {
		return magic[0] == 'i' && magic[1] == '3' && magic[2] == 'd' && magic[3] == 'm';
	}
	bool isCMPT() {
		return magic[0] == 'c' && magic[1] == 'm' && magic[2] == 'p' && magic[3] == 't';
	}
	bool isPNTS() {
		return magic[0] == 'p' && magic[1] == 'n' && magic[2] == 't' && magic[3] == 's';
	}
};


struct BatchTableHeader {
	uint32 batchTableJSONByteLength;
	uint32 batchTableBinaryByteLength;

	uint32 length() {
		return batchTableJSONByteLength + batchTableBinaryByteLength;
	}
};

struct FeatureTableHeader {
	uint32 featureTableJSONByteLength;
	uint32 featureTableBinaryByteLength;

	uint32 length() {
		return featureTableJSONByteLength + featureTableBinaryByteLength;
	}
};

struct BinaryGLTFHeader {
	unsigned char magic[4];
	uint32 version;
	uint32 length;
	uint32 contentLength;
	uint32 contentFormat;
};

struct Batch3DModelHeader {
	TileHeader general;
	BatchTableHeader batchTable;
	uint32 batchLength;

	uint32 getBatchStart() {
		return 24 /*headersize*/;
	}

	uint32 getGLTFStart() {
		return getBatchStart() + batchTable.length();
	}
};

struct Instanced3DModelHeader {
	TileHeader general;
	FeatureTableHeader featureTable;
	BatchTableHeader batchTable;
	uint32 gltfFormat;

	uint32 getFeatureStart() {
		return 32 /*headersize*/;
	}

	uint32 getBatchStart() {
		return getFeatureStart() + featureTable.length();
	}


	uint32 getGLTFStart() {
		return getBatchStart() + batchTable.length();
	}

	FString getpartAsString(int start,  int lenght) {
		char *ansiiData = new char[lenght + 1];
		memcpy(ansiiData, (void*)(((uint8*)this)+start), lenght); //Assumes bytesRead is always smaller than 1024 bytes
		ansiiData[lenght] = 0; //Add null terminator
		FString temp = ANSI_TO_TCHAR(ansiiData); //Convert to FString
		delete ansiiData;
		return temp;
	}
};

struct PointCloudHeader {
	TileHeader general;
	FeatureTableHeader featureTable;
	BatchTableHeader batchTable;


	uint32 getFeatureStart() {
		return 28 /*headersize*/;
	}

	uint32 getFeatureBinaryStart() {
		return getFeatureStart() + featureTable.featureTableJSONByteLength;
	}

	uint32 getBatchStart() {
		return getFeatureStart() + featureTable.length();
	}



	FString getpartAsString(int start, int lenght) {
		char *ansiiData = new char[lenght + 1];
		memcpy(ansiiData, (void*)(((uint8*)this) + start), lenght); //Assumes bytesRead is always smaller than 1024 bytes
		ansiiData[lenght] = 0; //Add null terminator
		FString temp = ANSI_TO_TCHAR(ansiiData); //Convert to FString
		delete ansiiData;
		return temp;
	}
};





/**
 * 
 */
UCLASS()
class MASTERTESTPROJECT_API ATilesetActor : public AActor
{
	GENERATED_BODY()

public:

	ATilesetActor(const FObjectInitializer& ObjectInitializer);

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Generation")
	FString host;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Generation")
	FString relativeURL;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "LOD")
	float LODtreshold = 0.1;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Texture")
	AStaticMeshActor  *TextureTest = NULL;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Texture")
	UTexture2D* tstTexture;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "PointcloudParent")
	USceneComponent* PointCloudParent;



	// Called when the game starts or when spawned
	virtual void BeginPlay() override;

	// Called every frame
	virtual void Tick(float DeltaSeconds) override;

	FTileset* parseTileset(FString JsonString, FString BaseURL);
	void parse3DTile(const TArray<uint8> data, FTile *tile);

	void PreReplication(IRepChangedPropertyTracker & ChangedPropertyTracker);

	//virtual bool ShouldTickIfViewportsOnly() override { return true; }

	void updateScreenSpaceError(FTile* current, double currentSSE, double constant, FVector CamLocation);

	UPROPERTY(replicated = false)
	FTileContent rootTileset;

private:
	bool doreplicateTielset = false;

	void parseTile(TSharedPtr<FJsonObject> json, FTile *targetTile, FTileset* parent);

	void parseBatched3DTile(const TArray<uint8> data, FTile *tile);
	void parseInstanced3DTile(const TArray<uint8> data, FTile *tile);
	void parsePointCloudTile(const TArray<uint8> data, FTile *tile);

	template <typename T> TArray<T> getAsArray(const uint8* data, int32 offset, int32 numberOfElements);
};

template<typename T>
inline TArray<T> ATilesetActor::getAsArray(const uint8 *data, int32 offset, int32 numberOfElements)
{
	T *ArrayPointer = (T*)(data + offset);
	TArray<T> result;
	result.Append(ArrayPointer, numberOfElements);
	return result;
}
	