// Fill out your copyright notice in the Description page of Project Settings.

#include "MasterTestProject.h"
#include "JsonReader.h"
#include "JsonObjectConverter.h"
#include "TileDownloader.h"
#include "ue4-assimp/ProceduralEntity.h"
#include "Tileset.h"
#include "PointCloudActor.h"
#include "Kismet/KismetMathLibrary.h"
#include "UnrealNetwork.h"


ATilesetActor::ATilesetActor(const FObjectInitializer& ObjectInitializer) {

	PrimaryActorTick.bCanEverTick = true;
	PrimaryActorTick.bStartWithTickEnabled = true;
	PrimaryActorTick.bAllowTickOnDedicatedServer = true;

}

void ATilesetActor::BeginPlay()
{
	Super::BeginPlay();

	//RootComponent = CreateDefaultSubobject<UStaticMeshComponent>("TESTMesh");

	//rootTileset = new FTileContent();
	rootTileset.url = relativeURL;

	UE_LOG(TILES, Error, TEXT("Tileset load %s: \"%s\" \"%s\""), GetWorld()->IsServer() ? TEXT("Server") : TEXT("Client") , *host, *relativeURL);
	//UE_LOG(TILES, Error, TEXT("Tileset load : %s %s"),  *relativeURL, *host);
	UTileDownloader *downloader = NewObject<UTileDownloader>(UTileDownloader::StaticClass());
	downloader->GetTileContent(this, &rootTileset, host);

}


void ATilesetActor::Tick(float DeltaSeconds)
{
	Super::Tick(DeltaSeconds);

	UWorld* const World = GetWorld();
	if (World && GEngine && GEngine->GameViewport //game init
		 && rootTileset.tileset) // tilset loaded
	{
		// viewport Cam pos and FOV
		APlayerController* Controller = World->GetFirstPlayerController();
		FVector CamLocation = Controller->PlayerCameraManager->GetCameraLocation();
		float FOV = Controller->PlayerCameraManager->GetFOVAngle();
		FVector2D ViewportSize = FVector2D(1, 1);
		GEngine->GameViewport->GetViewportSize(ViewportSize);

		double lambda = ViewportSize.X / FOV;

		
		FTile *tile = &rootTileset.tileset->root;

		//FVector ActorPos = FTransform(*tile->getAbsoluteTransform()).GetTranslation();
		//double dist = (CamLocation - ActorPos).Size();
		//
		////SSE http://folk.uio.no/inftt/Div/visualization2.pdf
		//double ScreenSpaceError1 = lambda * tile->geometricError / dist;

		updateScreenSpaceError(tile, 0, lambda, CamLocation);



		//SSE2 https://wiki.delphigl.com/index.php/Screenspace_Error
		double maxPixelError = 30.0f;
		double T = maxPixelError / ViewportSize.X;
		double C = FOV / T;
		double SSE2 = tile->geometricError * C;


		

	}

}

double FTile::getScreenSpaceError(double lambda, FVector CamLocation)
{
	FMatrix matrix = FMatrix();
	if (content.absoluteTileTranforms.Num() > 0) {
		matrix = content.absoluteTileTranforms[0];
	}
	else {
		matrix = *getAbsoluteTransform();
	}


	FVector ActorPos = FTransform(*this->getAbsoluteTransform()).GetTranslation();
	double dist = boundingVolume.getDistanceTo(CamLocation, matrix);
		
		(CamLocation - ActorPos).Size();

	if (content.url.EndsWith("i3d")) {
		

	}


	//SSE http://folk.uio.no/inftt/Div/visualization2.pdf
	double sse = lambda * this->geometricError / dist;
	//UE_LOG(TILES, Error, TEXT("sse = %f, geoE = %f Dist %f"), sse,  this->geometricError, dist);
	return sse;
}

void FTile::setVisible(bool visible)
{
	if (content.tiles.Num() == 0 && !content.tileset) {
		if (!content.loadingStarted && visible && !content.url.IsEmpty()) {

			
			if (parentTilset) {
				UTileDownloader *downloader = NewObject<UTileDownloader>(UTileDownloader::StaticClass());
				downloader->GetTileContent(parentTilset->parentActor, &content, parentTilset->absoluteURL, this);
				content.loadingStarted = true;
				UE_LOG(TILES, Warning, TEXT("Set Tile visible %s %s:"), parentTilset->parentActor->GetWorld()->IsServer() ? TEXT("Server") : TEXT("Client") , *content.url);
				
			}
			else {
				UE_LOG(TILES, Error, TEXT("Parent tilset NUll"));
			}
		

			//load
		}

		//maybe set parent visible untill loaded?
	}
	else if(content.tiles.Num() > 0) {
		for (AProceduralEntity* object : content.tiles) {
			object->setActorDisabled(!visible);
		}
	}
}


void ATilesetActor::updateScreenSpaceError(FTile* current, double currentSSE, double constant, FVector CamLocation)
{
	double ScreenSpaceError1 = current->getScreenSpaceError(constant, CamLocation);

	if (ScreenSpaceError1 > LODtreshold && current->children.Num() > 0) {
		for (FTile* child : current->children)
		{
			updateScreenSpaceError(child, 0, constant, CamLocation);
		}
		current->setVisible(true);//FIXME
	}
	else {
		current->setVisible(true);

		for (FTile* child : current->children)
		{
			child->setVisible(true);//FIXME
		}

		//UE_LOG(TILES, Error, TEXT("Set Visible %s SSError= %f"), *current->content.url, ScreenSpaceError1);
	}


	


	if (current->children.Num() == 0) {
		// set visible keine 
	}
	else {
		double SSEchildren = 0.0;
		for (FTile* child : current->children)
		{
			SSEchildren += child->getScreenSpaceError(constant, CamLocation);
		}

		//UE_LOG(TILES, Error, TEXT("SSError= %f,SSE2 Childs= %f"), ScreenSpaceError1, SSEchildren );
	}
	

}



FTileset* ATilesetActor::parseTileset(FString JsonString, FString BaseURL)
{

	//UE_LOG(TILES, Log, TEXT("JSON: %s"), *JsonString);
	//TODO error Handling
	TSharedPtr<FJsonObject> JsonParsed;
	TSharedRef<TJsonReader<TCHAR>> JsonReader = TJsonReaderFactory<TCHAR>::Create(JsonString);
	if (FJsonSerializer::Deserialize(JsonReader, JsonParsed))
	{
		FTileset *tileset = new FTileset();
		bool success = FJsonObjectConverter::JsonObjectToUStruct<FTileset>(JsonParsed.ToSharedRef(), tileset, 0, 0);
		tileset->absoluteURL = BaseURL;
		tileset->parentActor = this;

		TSharedPtr<FJsonObject> root = JsonParsed->GetObjectField("root");
		ATilesetActor::parseTile(root, &tileset->root, tileset);

		UE_LOG(TILES, Log, TEXT("Parsed Tileset %s: Success:%s Version:\"%s\" geometricError:%f"), *BaseURL, success ? TEXT("True") : TEXT("False"), *tileset->asset.version, tileset->geometricError);

		return tileset;
	}

	return nullptr;
}

void ATilesetActor::parseTile(TSharedPtr<FJsonObject> json, FTile *targetTile, FTileset* parent) {

	bool success = FJsonObjectConverter::JsonObjectToUStruct<FTile>(json.ToSharedRef(), targetTile, 0, 0);

	targetTile->parentTilset = parent;

	if (json->HasField("children")) {
		const TArray<TSharedPtr<FJsonValue>> children = json->GetArrayField("children");

		UE_LOG(TILES, Log, TEXT("Parsed Tile %s: Success:%s Version:\"%s\" Children: "), *targetTile->content.url, success ? TEXT("True") : TEXT("False"), *targetTile->content.url);

		for (TSharedPtr<FJsonValue> child : children)
		{
			FTile *newTile = new FTile();
			newTile->parent = targetTile;
			ATilesetActor::parseTile(child->AsObject(), newTile, parent);
			targetTile->children.Add(newTile);
		}
	}
	else {
		UE_LOG(TILES, Log, TEXT("Parsed Tile %s: Success:%s Version:\"%s\" no Childs "), *targetTile->content.url, success ? TEXT("True") : TEXT("False"), *targetTile->content.url);
	}



}

void ATilesetActor::parseBatched3DTile(const TArray<uint8> data, FTile * tile)
{
	Batch3DModelHeader *b3dmheader = (Batch3DModelHeader*)data.GetData();
	UE_LOG(TILES, Log, TEXT("Batch3DModel Batches: %d"), b3dmheader->batchLength);
	UWorld* const World = GetWorld();
	if (World)
	{
		AProceduralEntity *gltf = World->SpawnActor<AProceduralEntity>(AProceduralEntity::StaticClass());
		gltf->tile = *tile;
#if WITH_EDITOR
		gltf->SetFolderPath(FName(*("SpawnedGLTF/" + this->GetName())));
#endif
		gltf->AttachToActor(this, FAttachmentTransformRules::KeepWorldTransform);
		FMatrix leftToRight(FVector(-1, 0, 0), FVector(0, 0, 1), FVector(0, 1, 0), FVector(0, 0, 0));
		FMatrix globalTielTransform = leftToRight * *tile->getAbsoluteTransform();
		gltf->SetActorRelativeTransform(FTransform(globalTielTransform));
		int GLTFstart = b3dmheader->getGLTFStart();

		gltf->loadModel(&data.GetData()[GLTFstart], (data.Num() - GLTFstart));
		tile->content.tiles.Add(gltf);
		tile->content.absoluteTileTranforms.Add(globalTielTransform);
		tile->setVisible(false);
	}
}

void ATilesetActor::parseInstanced3DTile(const TArray<uint8> data, FTile * tile)
{
	Instanced3DModelHeader *i3dmheader = (Instanced3DModelHeader*)data.GetData();
	FString featureJSON = i3dmheader->getpartAsString(i3dmheader->getFeatureStart(), i3dmheader->featureTable.featureTableJSONByteLength);
	UE_LOG(TILES, Log, TEXT("Instanced3DModelHeader GLTFFormat %s FeatureTable JSON: %s"), i3dmheader->gltfFormat == 0 ? TEXT("URL") : TEXT("Binary GLTF"), *featureJSON);
	TSharedPtr<FJsonObject> JsonParsed;
	TSharedRef<TJsonReader<TCHAR>> JsonReader = TJsonReaderFactory<TCHAR>::Create(featureJSON);
	if (FJsonSerializer::Deserialize(JsonReader, JsonParsed))
	{
		int32 instances_length = JsonParsed->GetIntegerField("INSTANCES_LENGTH");// FIXME is uint32;

		UWorld* const World = GetWorld();
		if (World)
		{
			if (i3dmheader->gltfFormat == 1) {

				AProceduralEntity *gltf = World->SpawnActor<AProceduralEntity>(AProceduralEntity::StaticClass());
				gltf->tile = *tile;
#if WITH_EDITOR
				gltf->SetFolderPath(FName(*("SpawnedGLTF/" + this->GetName())));
#endif
				gltf->AttachToActor(this, FAttachmentTransformRules::KeepWorldTransform);
				gltf->SetActorRelativeTransform(FTransform(*tile->getAbsoluteTransform()));
				int GLTFstart = i3dmheader->getGLTFStart();

				gltf->loadModel(&data.GetData()[GLTFstart], (data.Num() - GLTFstart));

				//http://www.dirsig.org/docs/new/coordinates.html

				if (JsonParsed->HasField("POSITION")) {
					uint32 featureTableBinaryOffset = i3dmheader->getFeatureStart() + i3dmheader->featureTable.featureTableJSONByteLength;
					TSharedPtr<FJsonObject> binaryBodyReference = JsonParsed->GetObjectField("POSITION");
					int32 byteOffset = binaryBodyReference->GetIntegerField("byteOffset");

					TArray<FVector> positons = getAsArray<FVector>(data.GetData(), (int32)(featureTableBinaryOffset + byteOffset), instances_length);

					FString outputString = "[";

					for (FVector pos : positons)
					{
						outputString += "[" + pos.ToString() +"] ";
						AProceduralEntity* clone = gltf->clone();
						FMatrix instanceTranform = FMatrix();

						if (JsonParsed->GetBoolField("EAST_NORTH_UP")) {
							FVector up(pos);
							up.Normalize();
							FVector Z(0, 0, 1);
							FVector east = FVector::CrossProduct( Z, up );
							FVector north = FVector::CrossProduct(east, up);
							FMatrix NEU(north, up, east, pos);
							instanceTranform = NEU;
						}
						else {
							FTransform trans = FTransform(pos);
							instanceTranform = trans.ToMatrixWithScale();
							//TODO add rotation and ransform gltf -> tiles
						}

						clone->SetActorTransform(FTransform(instanceTranform));
						tile->content.absoluteTileTranforms.Add(instanceTranform);
						tile->content.tiles.Add(clone);
					}
					tile->content.tiles.Add(gltf);
					FTransform trans = FTransform(positons[0]);
					tile->content.absoluteTileTranforms.Add(trans.ToMatrixWithScale());
					gltf->SetActorTransform(trans);
					outputString += "]";
					UE_LOG(TILES, Log, TEXT("InStanced 3D Model Positions %s"), *outputString);
				}
				else {
					UE_LOG(TILES, Warning, TEXT("INStanced 3D Model Position Quantized not Supported in %s"), *tile->content.url);
				}
			}


		}
	}
	tile->setVisible(false);
}

void ATilesetActor::parsePointCloudTile(const TArray<uint8> data, FTile * tile)
{
	FDateTime begin = FDateTime::Now();
	PointCloudHeader *pntheader = (PointCloudHeader*)data.GetData();
	FString featureJSONString = pntheader->getpartAsString(pntheader->getFeatureStart(), pntheader->featureTable.featureTableJSONByteLength);
	UE_LOG(TILES, Log, TEXT("PointCloudHeader: FeatureTable JSON: %s"), *featureJSONString);
	TSharedPtr<FJsonObject> FeatureTableJSON;
	TSharedRef<TJsonReader<TCHAR>> JsonReader = TJsonReaderFactory<TCHAR>::Create(featureJSONString);
	UWorld* const World = GetWorld();
	if (World && FJsonSerializer::Deserialize(JsonReader, FeatureTableJSON))
	{
		int32 instances_length = FeatureTableJSON->GetIntegerField("POINTS_LENGTH");
		int32 TextureSize = FMath::FloorToInt(FMath::Sqrt(instances_length)) + 1;
		int32 fill = TextureSize*TextureSize - instances_length;
		int32 quadchainsize = pow(2, 20);

		if (!FeatureTableJSON->HasField("POSITION")) {
			UE_LOG(TILES, Error, TEXT("Pointcloud Quantized not supported Position Required"));
			return;
		}

		// POSITION = float32[3]
		uint32 featureTableBinaryOffset = pntheader->getFeatureStart() + pntheader->featureTable.featureTableJSONByteLength;
		TSharedPtr<FJsonObject> binaryBodyReference = FeatureTableJSON->GetObjectField("POSITION");
		int32 byteOffset = binaryBodyReference->GetIntegerField("byteOffset");
		uint8* start = (uint8*)(data.GetData() + pntheader->getFeatureBinaryStart() + byteOffset);

		// creat pooint texture
		TArray<FLinearColor> *Points = new TArray<FLinearColor>();
		uint8* pos = start;
		float minx, maxx, miny, maxy, minz, maxz;
		minx = miny = minz = TNumericLimits< float >::Max();	
		maxx = maxy = maxz = TNumericLimits< float >::Min();

		for (size_t i = 0; i < instances_length; i+=1)
		{
			float x = *((float*)pos);
			float y = *((float*)(pos + 4));
			float z = *((float*)(pos + 8));
	
			minx = FMath::Min(x, minx);
			maxx = FMath::Max(x, maxx);

			miny = FMath::Min(y, miny);
			maxy = FMath::Max(y, maxy);

			minz = FMath::Min(z, minz);
			maxz = FMath::Max(z, maxz);

			pos += 12;
		}

		pos = start;
		for (size_t i = 0; i < instances_length; i += 1)
		{
			float x = *((float*)pos);
			float y = *((float*)(pos + 4));
			float z = *((float*)(pos + 8));

			x = (x - minx) / (maxx - minx);
			y = (y - miny) / (maxy - miny);
			z = (z - minz) / (maxz - minz);

			FLinearColor point = FLinearColor(x,y, z, 1.0f);
			Points->Add(point);

			pos += 12;
		}
		for (size_t i = 0; i < fill; i += 1)
		{
			Points->Add(FLinearColor(0.0f, 0.0f, 0.0f, 0.0f));
		}
		UTexture2D* PointsTeture = UDynamicTextureUtilities::CreateTransientDynamicTexture(TextureSize, TextureSize, PF_A32B32G32R32F);
		UDynamicTextureUtilities::UpdateDynamicVectorTexture(*Points, PointsTeture);
		FVector min = FVector(minx, miny, minz);
		FVector max = FVector(maxx, maxy, maxz);
		FVector size = max - min;
		
		UE_LOG(TILES, Log, TEXT("Created Position Texture Number Points %d, TextureSize: %d, Fill: %d"), instances_length, TextureSize, fill);
		UE_LOG(TILES, Log, TEXT("Pointcloud Boundig Box: Min [%s] Max [%s] Size [%s]"), *min.ToString(), *max.ToString(), *size.ToString());


		

		UTexture2D* ColorTexture = nullptr;
		if (FeatureTableJSON->HasField("RGB")) {
			// RGB = uint8[3]
			featureTableBinaryOffset = pntheader->getFeatureStart() + pntheader->featureTable.featureTableJSONByteLength;
			binaryBodyReference = FeatureTableJSON->GetObjectField("RGB");
			byteOffset = binaryBodyReference->GetIntegerField("byteOffset");
			start = (uint8*)(data.GetData() + pntheader->getFeatureBinaryStart() + byteOffset);

			// creat pooint texture
			TArray<FLinearColor> *ColorRGB = new TArray<FLinearColor>();
			pos = start;
			for (size_t i = 0; i < instances_length; i += 1)
			{
				FLinearColor temp = FLinearColor((*pos) / 255.0, (*(pos + 1)) / 255.0, (*(pos + 2)) / 255.0, 1.0F);
				ColorRGB->Add(temp);
				pos += 3;
			}
			for (size_t i = 0; i < fill; i += 1)
			{
				ColorRGB->Add(FLinearColor(1.0f, 0.0f, 0.0f));
			}
			ColorTexture = UDynamicTextureUtilities::CreateTransientDynamicTexture(TextureSize, TextureSize, PF_A32B32G32R32F);
			UDynamicTextureUtilities::UpdateDynamicVectorTexture(*ColorRGB, ColorTexture);
			UE_LOG(TILES, Log, TEXT("Created Color Texture Number Points %d, TextureSize: %d, Fill: %d, byteOffset: %d"), instances_length, TextureSize, fill, byteOffset);
			
		}

		//Testing Texture
		if (TextureTest) {
			UMaterialInstanceDynamic *Material = UMaterialInstanceDynamic::Create(TextureTest->GetStaticMeshComponent()->GetMaterial(0), nullptr);
			Material->SetTextureParameterValue(FName("DynamicTexture"), PointsTeture);
			TextureTest->GetStaticMeshComponent()->SetMaterial(0,Material);
		}

		for (size_t i = 0; i < instances_length; i+=quadchainsize)
		{
			APointCloudActor *PointCloud = World->SpawnActor<APointCloudActor>(APointCloudActor::StaticClass());
			PointCloud->setPoints(PointsTeture, min, size, i);
#if WITH_EDITOR
		//	PointCloud->SetFolderPath(FName(*("SpawnedGLTF/" + this->GetName())));
#endif
			PointCloud->AttachToActor(this, FAttachmentTransformRules::KeepWorldTransform);

			FMatrix leftToRight(FVector(-1, 0, 0), FVector(0, 0, 1), FVector(0, 1, 0), FVector(0, 0, 0));
			FMatrix globalTielTransform = leftToRight*   *tile->getAbsoluteTransform();
			UE_LOG(TILES, Error, TEXT("Pointcloud Transform: %s"), *globalTielTransform.ToString());

			// do not, die quadchain muss bei 0/0/0 sein
			//PointCloud->SetActorRelativeTransform(FTransform(globalTielTransform));


			if (ColorTexture)
				PointCloud->setColors(ColorTexture);
		}
	}
	FDateTime end = FDateTime::Now();
	
	UE_LOG(TILES, Error, TEXT("Pointcloud Generation: %f"), (end - begin).GetTotalMilliseconds());
}

void ATilesetActor::parse3DTile(const TArray<uint8> data, FTile *tile)
{
	if (!tile) {

		char *ansiiData = new char[data.Num() + 2];
		memcpy(ansiiData, (void*)(data.GetData()), data.Num()); //Assumes bytesRead is always smaller than 1024 bytes
		ansiiData[data.Num()] = 0; //Add null terminator
		FString temp = ANSI_TO_TCHAR(ansiiData); //Convert to FString
		delete ansiiData;
		UE_LOG(TILES, Error, TEXT("Something went horribly wrong in processing a Tile: %s"), *temp);

		return;	
	}
	TileHeader *header = (TileHeader*) data.GetData();
	tile->content.content = data;

	UE_LOG(TILES, Log, TEXT("Shared Tile Header: Magic: %s Version %d lenght %d, array lenght %d "), *header->getMagicAsFString(), header->version, header->byteLength, data.Num());

	if (header->isB3DM()) {
		parseBatched3DTile(data, tile);
	}
	else if (header->isI3DM()) {
		parseInstanced3DTile(data, tile);
	}
	else if (header->isPNTS()) {
		parsePointCloudTile(data, tile);
	}
}




FMatrix * FTile::getMatrix()
{
	if (realtiveTransform == nullptr) {
		if (transform.Num() == 16) {
			FPlane X(transform[0], transform[1], transform[2], transform[3]);
			FPlane Y(transform[4], transform[5], transform[6], transform[7]);
			FPlane Z(transform[8], transform[9], transform[10], transform[11]);
			FPlane W(transform[12], transform[13], transform[14], transform[15]);

			//FMatrix(FPlane(), FPlane(), FPlane(), FPlane())
			realtiveTransform = new FMatrix(X, Y, Z, W);
		}
		else if (transform.Num() == 0) {
			realtiveTransform = new FMatrix();
			realtiveTransform->SetIdentity();
		}
		else {
			UE_LOG(TILES, Log, TEXT("Not allowed Transorm in %s ArrayLength: %d [0 or 16 allowed]"), *content.url, transform.Num());
		}
		//UE_LOG(TILES, Log, TEXT("%s Computed transform: %s"), *content.url, *realtiveTransform->ToString());
	}
	return realtiveTransform;
}

FMatrix * FTile::getAbsoluteTransform()
{
	if (!absoluteTransform) {
		FMatrix(absoluteTrans);
		absoluteTrans.SetIdentity();
		for (FTile *current = this; current != nullptr; current = current->parent) {
			absoluteTrans = *current->getMatrix() * absoluteTrans;
		}

		if (this->parentTilset != nullptr && this->parentTilset->parent != nullptr) {
			absoluteTrans = *this->parentTilset->parent->getAbsoluteTransform() * absoluteTrans;
		}

		absoluteTransform = new FMatrix(absoluteTrans);
	}

	return absoluteTransform;
}

float FBoundingVolume::getDistanceTo(FVector Position, FMatrix bouindingTransformation)
{
	FVector localPos = bouindingTransformation.Inverse().TransformFVector4(FVector4(Position,1));

	float distance = 0.0f;
	if (box.Num() == 12) {
		FVector Center = FVector(box[0], box[1], box[2]);
		//implicit Halflength
		FVector XAxis = FVector(box[3], box[4], box[5]);
		FVector YAxis = FVector(box[6], box[7], box[8]);
		FVector ZAxis = FVector(box[9], box[10], box[11]);

		//TODO FIXME is currently spher distance
		//UE_LOG(TILES, Error, TEXT("Box distance is not implemented"));
		double dist = (localPos - Center).Size();
		distance = dist -  FMath::Max3(XAxis.Size(), YAxis.Size(), ZAxis.Size());

	}
	else if (region.Num() == 6) {
		// WGS84
		//TODO FIXME is currently spher distance
		//UE_LOG(TILES, Error, TEXT("Region distance is not imyplemented"));
		FVector Center = FVector(0, 0, 6378137);
		double dist = (Position - Center).Size();
		distance = dist;
	}else if (sphere.Num() == 4) {
		double dist = (localPos - FVector(sphere[0], sphere[1], sphere[2])).Size();
		distance = dist - sphere[3];

	}
	else {
		UE_LOG(TILES, Error, TEXT("Not Allowed Tranformation"));
	}



	return distance;
}

void ATilesetActor::PreReplication(IRepChangedPropertyTracker & ChangedPropertyTracker)
{
	UE_LOG(TILES, Error, TEXT("test"));
	DOREPLIFETIME_ACTIVE_OVERRIDE(ATilesetActor, rootTileset, false);
}

void ATilesetActor::GetLifetimeReplicatedProps(TArray< FLifetimeProperty > & OutLifetimeProps) const
{
	DOREPLIFETIME(ATilesetActor, rootTileset);
}
