﻿
using UnityEngine;
using System.Collections;
using GamepadInput;

public class PlayerControl : MonoBehaviour
{
  private Rigidbody2D body2D;

  public Transform groundCheck;
  public LayerMask mask;

  // movement config
  public float pushForce = 5f;
  public float pushTime = 0.5f;
  public float gravity = -15f;
  public float runSpeed = 8f;
  public float groundDamping = 20f; // how fast do we change direction? higher means faster
  public float inAirDamping = 5f; //only horizontal
  public float inAirDampingVertical = 10f;
  public float targetJumpHeight = 2f;
  public float lastDashStart = float.NegativeInfinity;
  [HideInInspector]
  public Vector2 dashDirection = Vector2.right;
  [HideInInspector]
  public Vector2 currentDashDirection = Vector2.right;
  public float dashCompletionTime = 0.25f;
  public float dashStartSpeed = 20.0f;
  public float dashEndSpeed = 8.0f;
  public float dashCoolDown = 0.4f;
  public float dashVerticalScale = 0.5f;
  public float dashCollisionCoolDown = 1.5f;
  [HideInInspector]
  public float currentDashCoolDown = 0.0f;
  [HideInInspector]
  public bool hasLandedSinceLastDash = true;
  // public float jumpWaitTime = 2.0;

  [HideInInspector]
  public Vector2 pushSpeed = Vector2.zero;
  [HideInInspector]
  public float lastPushStart = float.NegativeInfinity;

  [HideInInspector]
  public float rawMovementDirection = 1;
  //[HideInInspector]
  public float normalizedHorizontalSpeed = 0;

  [HideInInspector]
  public Vector2 velocity;

  public int playerNumber = 1; //gibt an, ob es sich um player one, player two, etc. handelt. sollte nicht 0 sein;
  public PlayerInputMapping inputMapping;

  public Animator dashAnimator;

  void Start()
  {
    body2D = GetComponent<Rigidbody2D>();
  }

  public void init(int playerNumber)
  {
    this.playerNumber = playerNumber;

    //in der szene muss fuer jeden spieer so ein InputMapping"nr" sein:
    inputMapping = GameObject.Find("InputMapping" + playerNumber).GetComponent<PlayerInputMapping>();
    GetComponent<PlayerHealth>().init(playerNumber);
  }

  void OnCollisionEnter2D(Collision2D coll)
  {
    if (coll.collider.tag != "Player" || !isDashing())
      return;

    PlayerControl other = coll.collider.gameObject.GetComponent<PlayerControl>();
    other.setPushSpeed(currentDashDirection * pushForce);
    lastDashStart = Time.time - dashCompletionTime;
    currentDashCoolDown = dashCollisionCoolDown;

    //dash effect:
    dashAnimator.SetTrigger("play");
  }

  private void setPushSpeed(Vector2 pushSpeed)
  {
    this.pushSpeed = pushSpeed;
    this.lastPushStart = Time.time;
  }

  void FixedUpdate()
  {
    if (isGrounded())
      velocity.y = 0;

    updateDashDirection();

    if (canDash() && inputMapping.isDashPressed())
    {
      lastDashStart = Time.time;
      currentDashDirection = dashDirection;
      currentDashCoolDown = dashCoolDown;
      hasLandedSinceLastDash = false;
    }

    if (isDashing())
    {
      velocity = currentDashDirection * Mathf.Lerp(dashStartSpeed, dashEndSpeed, getDashTime() / dashCompletionTime);
    }
    else
    {
      bool grounded = isGrounded();
      if (grounded && !isDashing())
      {
        hasLandedSinceLastDash = true;
      }

      velocity = body2D.velocity;
      if (inputMapping.isRightPressed())
      {
        if (transform.localScale.x > 0f)
          transform.localScale = new Vector3(-transform.localScale.x, transform.localScale.y, transform.localScale.z);

        velocity.x = runSpeed;
      }
      else if (inputMapping.isLeftPressed())
      {
        if (transform.localScale.x < 0f)
          transform.localScale = new Vector3(-transform.localScale.x, transform.localScale.y, transform.localScale.z);
        velocity.x = -runSpeed;
      }

      if (inputMapping.isJumpPressed())
      {
        //to avoid DOUBLE JUMP
        if (grounded)
        {
          velocity.y = targetJumpHeight;
        }
      }

      // apply horizontal speed smoothing it
      var smoothedMovementFactor = isGrounded() ? groundDamping : inAirDamping; // how fast do we change direction?
      velocity.x = Mathf.Lerp(velocity.x, 0, Time.deltaTime * smoothedMovementFactor);

      float verticalSmoothedFactor = isGrounded() ? 0 : inAirDampingVertical;
      if (velocity.y > 0.0f) velocity.y = Mathf.Lerp(velocity.y, 0, Time.deltaTime * verticalSmoothedFactor);


      /*==========Power Ups // Bullet management ===*/
      if (inputMapping.isShootPressed())
      {
        Debug.Log("Shoot pressed. Can shoot: " + canShoot());
        if (canShoot())
        {
          shoot();
          coolDown();
        }
      }
    }

    float currentPushTime = Time.time - lastPushStart;
    Vector2 currentPushSpeed = Vector2.zero;
    if (0 <= currentPushTime && currentPushTime <= pushTime)
    {
      currentPushSpeed = Vector2.Lerp(pushSpeed, Vector2.zero, currentPushTime / pushTime);
    }

    //finally set the velocity:
    body2D.velocity = velocity + currentPushSpeed;
  }

  private bool isGrounded()
  {
    Vector2 playerPos = new Vector2(transform.position.x, transform.position.y);
    Vector2 groundPos = new Vector2(groundCheck.position.x, groundCheck.position.y);
    bool result = Physics2D.Linecast(playerPos, groundPos, mask);
    if (result)
    {
      Debug.DrawLine(playerPos, groundPos, Color.green, 0.5f, false);
    }
    else
    {
      Debug.DrawLine(playerPos, groundPos, Color.red, 0.5f, false);
    }

    return result;
  }

  private void updateDashDirection()
  {
    float horizontalDashDirection = 0.0f;
    if (inputMapping.isRightPressed())
    {
      horizontalDashDirection = 1.0f;
    }
    else if (inputMapping.isLeftPressed())
    {
      horizontalDashDirection = -1.0f;
    }

    float verticalDashDirection = 0.0f;
    if (inputMapping.isUpPressed())
    {
      verticalDashDirection = 1.0f;
    }
    else if (inputMapping.isDownPressed())
    {
      verticalDashDirection = -1.0f;
    }

    if (horizontalDashDirection != 0.0f || verticalDashDirection != 0.0f)
      dashDirection = new Vector2(horizontalDashDirection, dashVerticalScale * verticalDashDirection);
  }

  private float getDashTime()
  {
    return Time.time - lastDashStart;
  }

  private bool isDashing()
  {
    float dashTime = getDashTime();
    return 0 <= dashTime && dashTime < dashCompletionTime;
  }

  private bool canDash()
  {
    return hasLandedSinceLastDash && getDashTime() >= dashCompletionTime + currentDashCoolDown;
  }

  /*==========Power Ups // Bullet management ===*/

  public const int POWERUP_NONE = 0;
  public const int POWERUP_HEALINGBULLET = 1;
  public const int POWERUP_TRAPDESTROYER = 2;

  private int powerUpType = POWERUP_NONE;

  private bool coolingDown = false;

  public float coolDownTime = 2.0f;

  public GameObject healingBulletPrefab;
  public GameObject trapDestroyingBulletPrefab;

  public float spawnDistance = 1.0f;
  public void setPowerUpType(int type)
  {
    powerUpType = type;

  }

  private void shoot()
  {
    float dir = Mathf.Sign(transform.localScale.x);
    Vector2 pos = new Vector2(transform.position.x + dir * spawnDistance, transform.position.y);

    switch (powerUpType)
    {
      case POWERUP_HEALINGBULLET:
        GameObject bullet = Instantiate(healingBulletPrefab, pos, transform.rotation) as GameObject;
        bullet.GetComponent<Bullet>().shoot(new Vector2(dir, 0));
        break;
      case POWERUP_TRAPDESTROYER:
        GameObject trapbullet = Instantiate(trapDestroyingBulletPrefab, pos, transform.rotation) as GameObject;
        trapbullet.GetComponent<Bullet>().shoot(new Vector2(dir, 0));
        break;
      default:
        break;
    }
  }

  private void coolDown()
  {
    coolingDown = true;
    Invoke("coolDownOver", coolDownTime);
  }

  private void coolDownOver()
  {
    coolingDown = false;
  }

  private bool canShoot()
  {
    return (!coolingDown) && powerUpType != POWERUP_NONE;
  }

}
