using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using UnityEngine;

public class DeserializedLevelsLoader
{
  // Levels deserialized
  private DeserializedLevels deserializedLevels = null;

  private const string prefabsFolder = "Prefabs/";

  struct ItemStruct
  {
    public GameObject prefab;
    public float x;
    public float y;
    public float rot;
    public float scalex;
    public float scaley;
    public ItemChildStruct[] children;
  }

  struct ItemChildStruct
  {
    public string name;
    public float x;
    public float y;
    public float rot;
    public float scalex;
    public float scaley;
  }


  // Cache prefabs in prefabDict
  Dictionary<string, GameObject> prefabPool;

  // Cache all items with locations
  List<ItemStruct> sceneItemsList;

  Transform parentOfXmlItems;

  public const string xmlItemsGOName = "XmlItems";

  public void load()
  {
    deserializedLevels = XmlIO.LoadXml<DeserializedLevels>("Levels");
  }

  public int getLevelCount()
  {
    if (deserializedLevels == null)
      throw new System.InvalidOperationException();

    return deserializedLevels.levels.Length;
  }

  public void loadLevel(int levelIndex = 0)
  {
    if (deserializedLevels == null)
      throw new System.InvalidOperationException();

    if (levelIndex >= deserializedLevels.levels.Length)
      throw new System.ArgumentException();

    prefabPool = new Dictionary<string, GameObject>();
    sceneItemsList = new List<ItemStruct>();

    // if the XmlItems gameobject folder remained in the Hierarcy, then delete it
    while (GameObject.Find(xmlItemsGOName) != null)
      MonoBehaviour.DestroyImmediate(GameObject.Find(xmlItemsGOName));

    parentOfXmlItems = new GameObject(xmlItemsGOName).transform;

    DeserializedLevels.Level currentLevel = deserializedLevels.levels[levelIndex];
    sceneItemsList.Clear();


    // <Item prefab="Chair" x="1" y="10" rot="90" />
    foreach (DeserializedLevels.Item deserializedItem in currentLevel.items)
    {
      // caching prefabString i.e. "phone"
      string prefabString = deserializedItem.prefab;

      // if the prefab in the item XmlNode has not been loaded then add it to the prefabsDict dictionary,
      if (!prefabPool.ContainsKey(prefabString))
      {
        // load prefab
        GameObject prefabObject = Resources.Load(prefabsFolder + prefabString, typeof(GameObject)) as GameObject;

        // if unsuccesful, error message and jump to next in the foreach loop
        if (prefabObject == null)
        {
          Debug.LogError("Prefab \"" + prefabString + "\" does not exists.");
          continue;
        }

        // otherwise add to dictionary
        prefabPool.Add(prefabString, prefabObject);
      }

      ItemStruct item;
      item.prefab = prefabPool[prefabString];
      item.x = toFloatZeroIfNull(deserializedItem.x);
      item.y = toFloatZeroIfNull(deserializedItem.y);
      item.rot = toFloatZeroIfNull(deserializedItem.rot);
      item.scalex = toFloatOneIfNull(deserializedItem.scalex);
      item.scaley = toFloatOneIfNull(deserializedItem.scaley);

      DeserializedLevels.Item[] children;
      if (deserializedItem.children == null) children = new DeserializedLevels.Item[0];
      else children = deserializedItem.children;

      item.children = new ItemChildStruct[children.Length];
      int k = 0;
      foreach (DeserializedLevels.Item child in children)
      {
        item.children[k].name = child.prefab;
        item.children[k].x = toFloatZeroIfNull(child.x);
        item.children[k].y = toFloatZeroIfNull(child.y);
        item.children[k].rot = toFloatZeroIfNull(child.rot);
        item.children[k].scalex = toFloatOneIfNull(child.scalex);
        item.children[k].scaley = toFloatOneIfNull(child.scaley);
      }

      sceneItemsList.Add(item);
    }

    // Finally instantiate all items
    foreach (ItemStruct item in sceneItemsList)
    {

      // TODO load height coordinate from a directory
      GameObject newGameObject = MonoBehaviour.Instantiate(item.prefab) as GameObject;

      // set position
      setPos2D(newGameObject, new Vector2(item.x, item.y));

      // set rotation
      setRot2D(newGameObject, item.rot);

      setScale2D(newGameObject, item.scalex, item.scaley);

      foreach (Transform t in newGameObject.GetComponentsInChildren<Transform>())
      {
        if (!(t.parent == null) && (!t.parent.name.Equals("XmlItems")))
        {
          foreach (ItemChildStruct child in item.children) //Ja das ist wahrscheinlich Laufzeitmig nicht ideal, aber die Arrays haben eh nur eine Hand viele Elemente
          {
            if (t.name == child.name)
            {
              Debug.Log("Set child " + child.name);
              setPos2D(t.gameObject, new Vector2(child.x, child.y));
              setRot2D(t.gameObject, child.rot);
              setScale2D(t.gameObject, child.scalex, child.scaley);
            }
          }
        }
      }
      // set parent
      newGameObject.transform.parent = parentOfXmlItems;
    }
  }


  // DONE, these are only helper functions below

  // if no value then return zero or one, otherwise convert to float
  float toFloatZeroIfNull(string value) { return value == null ? 0 : float.Parse(value); }
  float toFloatOneIfNull(string value) { return value == null ? 1 : float.Parse(value); }

  void setPos2D(GameObject g, Vector2 pos)
  {
    g.transform.position = new Vector2(
        pos.x,
        pos.y
        );
  }

  void setRot2D(GameObject g, float rot)
  {
    //Debug.Log("Set rotation of " + g.name + " to " + rot);
    Quaternion rotation = Quaternion.identity;
    rotation.eulerAngles = new Vector3(0, 0, rot);
    g.transform.localRotation = rotation;
  }

  void setScale2D(GameObject g, float scaleX, float scaleY)
  {
    g.transform.localScale = new Vector2(scaleX, scaleY);
  }
}
