#include "myglwidget.h"
#include "controller.h"
#include "scenegraph.h"

//QGLShaderProgram *MyGLWidget::phongShader;

MyGLWidget::MyGLWidget(int index, Controller* c, bool isPerspective, QQuaternion initialRotation, QWidget *parent)
    : QGLWidget(QGLFormat(QGL::SampleBuffers), parent)
{
    this->index = index;
    MyGLWidget::modeCamera = true;
    tesselation = 1;
    cameraRotation = initialRotation;
    cameraStartRotation = initialRotation;
    camRotCenter = QVector3D(0,0,0);
    cameraZoom = cameraZoomDefault;
    controller = c;
    this->isPerspective = isPerspective;
    this->isFocused = isPerspective; //start with focus on perspective view

    scene = controller->getSceneGraph();
    scene->registerView(this);
}

MyGLWidget::~MyGLWidget()
{
    /*http://doc.qt.io/qt-4.8/qt-opengl-hellogl-example.html :
    * We use a destructor to ensure that any OpenGL-specific
    * data structures are deleted when the widget is no longer
    * needed (although in this case nothing needs cleaning up).
    */
    //In helloGL example: nothing needs cleaning up.
}

//0 = perspective, 1 = front, 2 = left, 3 = top. Hardcoded on construction.
int MyGLWidget::getIndex(){
    return index;
}

/*
 * From Example
 * We provide size hint functions to ensure that
 * the widgetis shown at a reasonable size:
 */
QSize MyGLWidget::minimumSizeHint() const
{
    return QSize(50, 50);
}

/*
 * From Example
 * We provide size hint functions to ensure that
 * the widgetis shown at a reasonable size:
 */
QSize MyGLWidget::sizeHint() const
{
    return QSize(600, 400);
}


void MyGLWidget::initializeGL()
{
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glEnable(GL_DEPTH_TEST);
    qDebug("InitGPL");
//    if (!phongShader){
        qDebug("Init shader stuff");
        initShaderStuff();
//    }else {
//        qDebug() << " phong not null: " << phongShader;
//    }


//    glEnable(GL_CULL_FACE);//backface culling.
//    glCullFace(GL_BACK);//backface culling

//    glEnable(GL_COLOR_MATERIAL); //Habe ich für glColor3f gebraucht. Wieder raus für Material. //In order to use both color and lighting you must activate openGL color materials

    //4.1.1
    /*
     * The initial state of the OpenGL state machine is defined in
initializeGL(). Furthermore, you can set all OpenGL parameters
that will not change during program execution here as well, for
example, the background color.
     */

    glEnable(GL_LIGHTING);//enable lighting.
    glEnable(GL_LIGHT0);//enable an OpenGL light
 /*
  * https://msdn.microsoft.com/en-us/library/windows/desktop/dd373578%28v=vs.85%29.aspx
  * The glLightfv function returns light source parameter values.
  * GL_POSITION:
  * The params parameter contains four floating-point values
  * that specify the position of the light in homogeneous object
  * coordinates. Both integer and floating-point values are
  * mapped directly. Neither integer nor floating-point values
  * are clamped.
 */
    static GLfloat lightPosition[4] = {0.5, 0.0, 2.0, 1.0};
    glLightfv(GL_LIGHT0,GL_POSITION,lightPosition);//set Position of light source.
}

void MyGLWidget::initShaderStuff(){
    //Phong Shader:
    QGLShader* vert = new QGLShader(QGLShader::Vertex);
    qDebug() << "new vert shader";
    QGLShader* frag = new QGLShader(QGLShader::Fragment);
    qDebug() << "new frag shader";

    bool e = QFile::exists(":/phong.vert");

    qDebug() << "vert exists" << e;

    bool successVertex = vert->compileSourceFile(":/phong.vert");
    if (!successVertex)qDebug() << "Vertex compilation failed.";
    else qDebug() << "Vertex compiled.";
    bool successFragment = frag->compileSourceFile(":/phong.frag");
    if (!successFragment) qDebug() << "Frag failed";
    else qDebug() << "frag success";

    phongShader = new QGLShaderProgram(this);
    phongShader->addShader(vert);
    phongShader->addShader(frag);
    phongShader->link();


    shadeFlatSlot();//set up flat shading
    qDebug() << "FlatSlot setup.";
}

void MyGLWidget::paintGL(){
    qDebug("paint gl");
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glLoadIdentity();

    //cam rotation
    QMatrix4x4 camRot = QMatrix4x4();
    camRot.rotate(cameraRotation);
    QMatrix4x4 zoomCameraMatrix = QMatrix4x4(1,0,0,0,0,1,0,0,0,0,1,cameraZoom,0,0,0,1);

    //So scheint das zumindest für die Kamera zu funktionieren:
    //1. ins Rotationszentrum verschieben - Wenn Rotationszentrum geändert wird, schon rotation berücksichtigen
    //2. Rotieren
    //3. Zoomen
    glMultMatrixf(zoomCameraMatrix.data());
    glMultMatrixf(camRot.data());
    glTranslatef(camRotCenter.x(),camRotCenter.y(),camRotCenter.z());

//    scene->drawAll();

    const QList<RigidBodyTransformation*> graph = scene->getGraph();
    QList<RigidBodyTransformation*>::const_iterator i;
    for (i = graph.begin(); i != graph.end(); ++i){

        //Set up Transformation:

        //rotation
        const QQuaternion rot = (*i)->getRotation();
        QMatrix4x4 m = QMatrix4x4();
        m.rotate(rot);

        //translation
        const QVector3D pos = (*i)->getTranslation();
        QMatrix4x4 translateRot1 = QMatrix4x4(1,0,0,pos.x(),0,1,0,pos.y(),0,0,1,pos.z(),0,0,0,1);

        //stack matrices
        glPushMatrix();
        glMultMatrixf(translateRot1.data());//Punkte zurück schieben damit rotation um qube zentrum ist
        glMultMatrixf(m.data());

        //draw primitive
        Primitive* p = (*i)->getChild();
        int tesselation = p->getTesselation();
        Primitive::Type type = p->getType();
        qDebug() << "draw type " << type;
        if (type == Primitive::Type::SPHERE){
            drawSphere(tesselation);
        }else if (type == Primitive::Type::BOX){
            drawBox(tesselation);
        }else if (type == Primitive::Type::CYLINDER){
            drawCylinder(tesselation);
        }else if (type == Primitive::Type::CONE){
            drawCone(tesselation);
        } else if (type == Primitive::Type::TORUS){
            drawTorus(tesselation);
        }else {
            qDebug() << "Weird Type." << type;
        }

        //reset transformation for next primitive
        glPopMatrix();
    }

    //has to be down here or will be overpainted by above code
    if (isFocused) highlightViewport();

}

void MyGLWidget::setFocused(bool isFocused){
    this->isFocused = isFocused;
    updateGL();
}

void MyGLWidget::highlightViewport(){
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-1,1,-1,1, -1, 1);//to draw yellow rectangle in screen coordinates

    GLfloat specularColor[] = {1,1,1};
    GLfloat shininess[] = {128};//specular exponent
    glMaterialfv(GL_FRONT,GL_SPECULAR,specularColor);
    glMaterialfv(GL_FRONT,GL_SHININESS,shininess);

    GLfloat yellow[] = {1,1,0};
    glMaterialfv(GL_FRONT,GL_DIFFUSE,yellow);
    glNormal3f(0,0,1);

    glDisable(GL_DEPTH_TEST);
    glLineWidth(15.0f);
    glBegin(GL_LINE_LOOP);
    glVertex2f(1,1);
    glVertex2f(1,-1);
    glVertex2f(-1,-1);
    glVertex2f(-1,1);
    glEnd();
    glEnable(GL_DEPTH_TEST);
    resetProjectionMatrix();
}

void MyGLWidget::resizeGL(int width, int height){
    windowWidth = width;
    windowHeight = height;
        qDebug() << "Resize to " << width << "," << height << ", Perspective: " << (double)windowWidth / (double)windowHeight;
    glViewport(0,0,width,height);

    resetProjectionMatrix();

   controller->setScreenScenter(QPoint((double)width/2.0,(double)height/2.0));//lieber zu oft casten bevors nicht tut...
}

void MyGLWidget::resetProjectionMatrix(){

    double p = (double)windowWidth / (double)windowHeight;
    glMatrixMode(GL_PROJECTION);//from HelloGL example
    glLoadIdentity();

    if (isPerspective){
        gluPerspective(45.0,p,0.1,1000);//set vof to 45°
    } else{
        glOrtho(-2*p,2*p,-2,2,1,1000);
    }

    glMatrixMode(GL_MODELVIEW);

}

/*User Input*/

void MyGLWidget::mousePressEvent(QMouseEvent *event){
    controller->processMousePressEvent(event,this);
}

void MyGLWidget:: mouseMoveEvent(QMouseEvent *event){
    controller->processMouseMoveEvent(event,this);
}

void MyGLWidget::keyPressEvent(QKeyEvent* event){
    if (event->key() == Qt::Key_Control){
        controller->setCtrlPressed(true);
    }
}

void MyGLWidget::keyReleaseEvent(QKeyEvent *event){
    if (event->key() == Qt::Key_Control){
        controller->setCtrlPressed(false);
        qDebug("ctrl released");
    }
}

//4.1.2 User Input: Zooming
void MyGLWidget::wheelEvent(QWheelEvent *event){//I don't want to extract this little code to the controller...
    cameraZoom += 0.01*event->delta();//0.01 seemed to be a good factor
    qDebug() << "Cam zoom delta " << event->delta() << " stored " << cameraZoom;
    updateGL();
}


/* View Modification: Camera*/
void MyGLWidget::translateCamera(double xDiff, double yDiff, double zDiff)
{
    //in order to "move parallel to image plane", we have to rotate the image coordinates xDiff and yDiff by our current camera rotation.
    camRotCenter += cameraRotation.conjugate().rotatedVector(QVector3D(xDiff,yDiff,zDiff));
    updateGL();
}

void MyGLWidget::rotateCamera(QQuaternion diff){
    if (isPerspective){
        cameraRotation = diff*cameraRotation;
        cameraRotation.normalize();
        updateGL();
    }
}

QQuaternion MyGLWidget::getCameraRotation(){
    return cameraRotation;
}


void MyGLWidget::resetCamera(){
    camRotCenter = QVector3D(0,0,0);
    cameraZoom = cameraZoomDefault;
    cameraRotation = cameraStartRotation;
    updateGL();
}


/*Notification for changed model*/
void MyGLWidget::modelChanged(){
    qDebug("update view");
    updateGL();
}

void MyGLWidget::drawSphere(int tesselation){
    GLfloat specularColor[] = { 0.3, 0.5, 0.5 };
    GLfloat shininess[] = { 120 };//specular exponent
    glMaterialfv(GL_FRONT, GL_SPECULAR, specularColor);
    glMaterialfv(GL_FRONT, GL_SHININESS, shininess);

    GLfloat sphereColor[] = { 0, 0.8, 0.8 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, sphereColor);
    glNormal3f(0, 0, 1);

    glBegin(GL_LINE_LOOP);
    GLUquadric* q = gluNewQuadric();
//    gluQuadricDrawStyle(q, GLU_FILL);
    gluSphere(q, 0.5, tesselation, tesselation);
    glEnd();
}

void MyGLWidget::drawBox(int tesselation){
    //tesselation
    double qubeWidth = 1.0;
    double dist = qubeWidth / (double)tesselation;

    //4.1.1 Unit Qube + Tesselation + Diffuse Material


    GLfloat specularColor[] = { 1, 1, 1 };
    GLfloat shininess[] = { 128 };//specular exponent
    glMaterialfv(GL_FRONT, GL_SPECULAR, specularColor);
    glMaterialfv(GL_FRONT, GL_SHININESS, shininess);

    //vorne // z-Achse
    //    glColor3f(0,0,1);//blau //alt und doof
    GLfloat blue[] = { 0, 0, 1 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, blue);
    glNormal3f(0, 0, 1);
    glBegin(GL_QUADS);
    for (int i = 0; i < tesselation; i++){
        for (int j = 0; j < tesselation; j++){
            glVertex3f(0.5 - dist*i, 0.5 - dist*j, 0.5);
            glVertex3f(0.5 - dist*(i + 1), 0.5 - dist*j, 0.5);
            glVertex3f(0.5 - dist*(i + 1), 0.5 - dist*(j + 1), 0.5);
            glVertex3f(0.5 - dist*i, 0.5 - dist*(j + 1), 0.5);
        }
    }
    glEnd();

    //oben // y-Achse
    GLfloat green[] = { 0, 1, 0 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, green);
    glNormal3f(0, 1, 0);
    glBegin(GL_QUADS);
    for (int i = 0; i < tesselation; i++){
        for (int j = 0; j < tesselation; j++){
            glVertex3f(0.5 - dist*(i + 1), 0.5, 0.5 - dist*(j + 1));
            glVertex3f(0.5 - dist*(i + 1), 0.5, 0.5 - dist*j);
            glVertex3f(0.5 - dist*i, 0.5, 0.5 - dist*j);
            glVertex3f(0.5 - dist*i, 0.5, 0.5 - dist*(j + 1));
        }
    }
    glEnd();

    //rechts //x-Achse
    GLfloat red[] = { 1, 0, 0 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, red);
    glNormal3f(1, 0, 0);
    glBegin(GL_QUADS);
    for (int i = 0; i < tesselation; i++){
        for (int j = 0; j < tesselation; j++){
            glVertex3f(0.5, 0.5 - dist*i, 0.5 - dist*j);
            glVertex3f(0.5, 0.5 - dist*(i + 1), 0.5 - dist*j);
            glVertex3f(0.5, 0.5 - dist*(i + 1), 0.5 - dist*(j + 1));
            glVertex3f(0.5, 0.5 - dist*i, 0.5 - dist*(j + 1));
        }
    }
    glEnd();

    //hinten // z-Achse
    GLfloat yellow[] = { 1, 0.9f, 0 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, yellow);
    glNormal3f(0, 0, -1);
    glBegin(GL_QUADS);
    for (int i = 0; i < tesselation; i++){
        for (int j = 0; j < tesselation; j++){
            glVertex3f(0.5 - dist*i, 0.5 - dist*(j + 1), -0.5);//4
            glVertex3f(0.5 - dist*(i + 1), 0.5 - dist*(j + 1), -0.5);//3
            glVertex3f(0.5 - dist*(i + 1), 0.5 - dist*j, -0.5);//2
            glVertex3f(0.5 - dist*i, 0.5 - dist*j, -0.5);//1
        }
    }
    glEnd();

    //links
    GLfloat cyan[] = { 0, 1, 1 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, cyan);
    glNormal3f(-1, 0, 0);
    glBegin(GL_QUADS);
    for (int i = 0; i < tesselation; i++){
        for (int j = 0; j < tesselation; j++){
            glVertex3f(-0.5, 0.5 - dist*i, 0.5 - dist*(j + 1));
            glVertex3f(-0.5, 0.5 - dist*(i + 1), 0.5 - dist*(j + 1));
            glVertex3f(-0.5, 0.5 - dist*(i + 1), 0.5 - dist*j);
            glVertex3f(-0.5, 0.5 - dist*i, 0.5 - dist*j);
        }
    }
    glEnd();

    //unten //-y
    GLfloat magenta[] = { 1, 0, 1 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, magenta);
    glNormal3f(0, -1, 0);
    glBegin(GL_QUADS);
    for (int i = 0; i < tesselation; i++){
        for (int j = 0; j < tesselation; j++){
            glVertex3f(0.5 - dist*i, -0.5, 0.5 - dist*(j + 1));
            glVertex3f(0.5 - dist*i, -0.5, 0.5 - dist*j);
            glVertex3f(0.5 - dist*(i + 1), -0.5, 0.5 - dist*j);
            glVertex3f(0.5 - dist*(i + 1), -0.5, 0.5 - dist*(j + 1));
        }
    }
    glEnd();
}

void MyGLWidget::drawCylinder(int tesselation){
    qDebug("draw cylinder");
    GLfloat specularColor[] = { 0.3, 0.5, 0.5 };
    GLfloat shininess[] = { 120 };//specular exponent
    glMaterialfv(GL_FRONT, GL_SPECULAR, specularColor);
    glMaterialfv(GL_FRONT, GL_SHININESS, shininess);

    GLfloat cylinderColor[] = { 0.8,0.2,0.5 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, cylinderColor);
    glNormal3f(0, 0, 1);

    glBegin(GL_LINE_LOOP);
    GLUquadric* quadric = gluNewQuadric();
    gluQuadricDrawStyle(quadric, GLU_FILL);
    gluCylinder(quadric,0.5,0.5,1,tesselation,tesselation);
    gluQuadricOrientation(quadric,GLU_INSIDE);
    gluDisk( quadric, 0.0, 0.5, tesselation, 1);
    glTranslatef( 0,0,1 );

    gluQuadricOrientation(quadric,GLU_OUTSIDE);
    gluDisk( quadric, 0.0, 0.5, tesselation, 1);
    gluDeleteQuadric(quadric);
    glEnd();
}

void MyGLWidget::drawCone(int tesselation){
    qDebug("draw cone");
    GLfloat specularColor[] = { 0.3, 0.5, 0.5 };
    GLfloat shininess[] = { 120 };//specular exponent
    glMaterialfv(GL_FRONT, GL_SPECULAR, specularColor);
    glMaterialfv(GL_FRONT, GL_SHININESS, shininess);

    GLfloat coneColor[] = { 0.8,0.8,0 };
    glMaterialfv(GL_FRONT, GL_DIFFUSE, coneColor);
    glNormal3f(0, 0, 1);

    glBegin(GL_LINE_LOOP);
    GLUquadric* quadric = gluNewQuadric();
    gluQuadricDrawStyle(quadric, GLU_FILL);
    gluCylinder(quadric,0.5,0,1,tesselation,1);
    gluQuadricOrientation(quadric,GLU_INSIDE);
    gluDisk( quadric, 0.0, 0.5, tesselation, 1);
    gluQuadricOrientation(quadric,GLU_OUTSIDE);
    gluDeleteQuadric(quadric);
    glEnd();
}

void MyGLWidget::drawTorus(int tesselation){

}


//4.1.1 slots for shading modes

void MyGLWidget::shadeWireframeSlot(){
    qDebug() << "Wireframe selected";

    phongShader->release();
    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);//Wireframe looks way cooler with both faces on
    updateGL();
}

void MyGLWidget::shadeFlatSlot(){
    qDebug() << "Flat selected";
    phongShader->release();
    glShadeModel(GL_FLAT);
    glPolygonMode(GL_FRONT,GL_FILL);
    updateGL();
}

void MyGLWidget::shadeGouraudSlot(){
    qDebug() << "Gouraud selected";
    phongShader->release();
    glShadeModel(GL_SMOOTH);
    glPolygonMode(GL_FRONT,GL_FILL);
    updateGL();
}

void MyGLWidget::shadePhongSlot(){
    qDebug() << "Phong selected";
    glPolygonMode(GL_FRONT,GL_FILL);
    phongShader->bind();
    updateGL();
}


